import React, { createContext, useContext, useState, useEffect } from "react";
import { type Salad } from "@shared/schema";

interface CartItem extends Salad {
  quantity: number;
}

interface CartContextType {
  items: CartItem[];
  addToCart: (salad: Salad) => void;
  removeFromCart: (saladId: number) => void;
  updateQuantity: (saladId: number, delta: number) => void;
  clearCart: () => void;
  totalItems: number;
  totalPrice: number;
}

const CartContext = createContext<CartContextType | undefined>(undefined);

export function CartProvider({ children }: { children: React.ReactNode }) {
  const [items, setItems] = useState<CartItem[]>(() => {
    const saved = localStorage.getItem("daily_bowl_cart");
    return saved ? JSON.parse(saved) : [];
  });

  useEffect(() => {
    localStorage.setItem("daily_bowl_cart", JSON.stringify(items));
  }, [items]);

  const addToCart = (salad: Salad) => {
    setItems((prev) => {
      const existing = prev.find((item) => item.id === salad.id);
      if (existing) {
        return prev.map((item) =>
          item.id === salad.id ? { ...item, quantity: item.quantity + 1 } : item
        );
      }
      return [...prev, { ...salad, quantity: 1 }];
    });
  };

  const removeFromCart = (saladId: number) => {
    setItems((prev) => prev.filter((item) => item.id !== saladId));
  };

  const updateQuantity = (saladId: number, delta: number) => {
    setItems((prev) =>
      prev.map((item) => {
        if (item.id === saladId) {
          const newQty = Math.max(0, item.quantity + delta);
          return { ...item, quantity: newQty };
        }
        return item;
      }).filter(item => item.quantity > 0)
    );
  };

  const clearCart = () => setItems([]);

  const totalItems = items.reduce((sum, item) => sum + item.quantity, 0);
  const totalPrice = items.reduce((sum, item) => sum + item.price * item.quantity, 0);

  return (
    <CartContext.Provider value={{ items, addToCart, removeFromCart, updateQuantity, clearCart, totalItems, totalPrice }}>
      {children}
    </CartContext.Provider>
  );
}

export function useCart() {
  const context = useContext(CartContext);
  if (!context) throw new Error("useCart must be used within a CartProvider");
  return context;
}
