import React, { createContext, useContext, useState, ReactNode } from 'react';

type Language = 'en' | 'mr';

interface I18nContextType {
  language: Language;
  setLanguage: (lang: Language) => void;
  t: (key: string) => string;
}

const translations: Record<string, Record<Language, string>> = {
  "nav.home": { en: "Home", mr: "होम" },
  "nav.menu": { en: "Menu", mr: "मेनू" },
  "nav.subscriptions": { en: "Subscriptions", mr: "सदस्यता" },
  "nav.franchises": { en: "Franchises", mr: "फ्रँचायझ" },
  "nav.about": { en: "About", mr: "आमच्याबद्दल" },
  "nav.contact": { en: "Contact", mr: "संपर्क" },
  
  "hero.title": { en: "Fresh. Healthy. Delicious.", mr: "ताजं. पौष्टिक. चवदार." },
  "hero.subtitle": { en: "Experience the taste of health with Daily Bowl.", mr: "डेली बाउलसोबत आरोग्याची चव अनुभवा." },
  "hero.cta": { en: "Order Now", mr: "ऑर्डर करा" },
  
  "why.title": { en: "Why Daily Bowl?", mr: "डेली बाउल का?" },
  "why.fresh": { en: "Fresh Ingredients", mr: "ताजे घटक" },
  "why.fresh.desc": { en: "Farm-to-bowl freshness in every bite.", mr: "प्रत्येक घासात शेतातील ताजेपणा." },
  "why.health": { en: "Health Focused", mr: "आरोग्यावर लक्ष" },
  "why.health.desc": { en: "Curated for weight loss, muscle gain & detox.", mr: "वजन कमी करणे, स्नायू वाढवणे आणि डिटॉक्ससाठी खास." },
  "why.taste": { en: "Authentic Taste", mr: "खारी चव" },
  "why.taste.desc": { en: "Recipes that delight your taste buds.", mr: "तुमच्या जिभेला तृप्त करणारी पाककृती." },
  
  "menu.title": { en: "Our Healthy Menu", mr: "आमचा पौष्टिक मेनू" },
  "menu.price": { en: "Price", mr: "किंमत" },
  "menu.add": { en: "Add to Cart", mr: "कार्टमध्ये टाका" },
  "menu.details": { en: "View Details", mr: "तपशील पहा" },
  
  "nutrition.calories": { en: "Calories", mr: "कॅलरीज" },
  "nutrition.protein": { en: "Protein", mr: "प्रथिने" },
  "nutrition.carbs": { en: "Carbs", mr: "कार्बोहायड्रेट्स" },
  "nutrition.fat": { en: "Fat", mr: "चरबी" },
  "nutrition.ingredients": { en: "Ingredients", mr: "साहित्य" },
  
  "contact.title": { en: "Get in Touch", mr: "संपर्क साधा" },
  "contact.name": { en: "Your Name", mr: "तुमचे नाव" },
  "contact.phone": { en: "Phone Number", mr: "फोन नंबर" },
  "contact.message": { en: "Message", mr: "संदेश" },
  "contact.submit": { en: "Send Message", mr: "संदेश पाठवा" },
  "contact.sending": { en: "Sending...", mr: "पाठवत आहे..." },

  "brand.ambassador": { en: "Brand Ambassador", mr: "ब्रँड ॲम्बेसेडर" },
  "brand.endorsement": { en: "Endorsed by", mr: "प्रायोजित" },
};

const I18nContext = createContext<I18nContextType | undefined>(undefined);

export function I18nProvider({ children }: { children: ReactNode }) {
  const [language, setLanguage] = useState<Language>('en');

  const t = (key: string) => {
    return translations[key]?.[language] || key;
  };

  return (
    <I18nContext.Provider value={{ language, setLanguage, t }}>
      {children}
    </I18nContext.Provider>
  );
}

export function useI18n() {
  const context = useContext(I18nContext);
  if (!context) {
    throw new Error("useI18n must be used within an I18nProvider");
  }
  return context;
}
