import { useSubscriptions } from "@/hooks/use-subscriptions";
import { useI18n } from "@/lib/i18n";
import { Loader2, Check } from "lucide-react";
import { type Subscription } from "@shared/schema";

const DEFAULT_IMAGE = "https://images.unsplash.com/photo-1546069901-ba9599a7e63c?w=800&q=80&auto=format&fit=crop";

export default function Subscriptions() {
  const { data: subscriptions, isLoading, error } = useSubscriptions();
  const { t, language } = useI18n();

  if (isLoading) {
    return (
      <div className="flex items-center justify-center min-h-[60vh]">
        <Loader2 className="w-12 h-12 text-primary animate-spin" />
      </div>
    );
  }

  if (error) {
    return (
      <div className="flex flex-col items-center justify-center min-h-[60vh] text-center px-4">
        <h3 className="text-2xl font-bold text-gray-800 mb-2">Oops! Something went wrong.</h3>
        <p className="text-gray-500">Could not load subscriptions. Please try again later.</p>
      </div>
    );
  }

  // Separate weekly and monthly subscriptions
  const weeklyPlans = subscriptions?.filter(s => s.type === 'weekly') || [];
  const monthlyPlans = subscriptions?.filter(s => s.type === 'monthly') || [];

  const SubscriptionCard = ({ plan }: { plan: Subscription }) => (
    <div className="bg-white rounded-3xl shadow-lg shadow-gray-200/50 overflow-hidden border border-gray-100 hover:shadow-xl hover:border-primary/20 transition-all duration-300 flex flex-col h-full group">
      {/* Image Area */}
      <div className="relative h-48 overflow-hidden bg-gray-100">
        <img 
          src={plan.imageUrl || DEFAULT_IMAGE}
          alt={plan.name}
          onError={(e) => {
            (e.target as HTMLImageElement).src = DEFAULT_IMAGE;
          }}
          className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-110"
          loading="lazy"
        />
      </div>

      {/* Content Area */}
      <div className="p-8 flex flex-col flex-grow">
        <div className="mb-2">
          <h3 className="font-display font-bold text-2xl text-gray-900 group-hover:text-primary transition-colors mb-2">
            {language === 'mr' && plan.marathiName ? plan.marathiName : plan.name}
          </h3>
          <span className="inline-block px-3 py-1 rounded-full bg-primary/10 text-primary text-xs font-bold uppercase tracking-wide">
            {plan.type === 'weekly' ? 'Weekly' : 'Monthly'}
          </span>
        </div>

        <p className="text-gray-600 text-sm mb-6 flex-grow">
          {language === 'mr' && plan.marathiDescription ? plan.marathiDescription : plan.description}
        </p>

        {/* Plan Details */}
        <div className="bg-gray-50 rounded-2xl p-4 mb-6 space-y-3 border border-gray-100">
          <div className="flex justify-between items-center">
            <span className="text-gray-600 text-sm font-medium">Salads per week</span>
            <span className="font-display font-bold text-lg text-primary">{plan.saladsPerWeek}</span>
          </div>
          <div className="border-t border-gray-200 pt-3">
            <div className="flex items-baseline gap-2">
              <span className="text-gray-500 text-sm line-through">₹{plan.basePrice}</span>
              <span className="font-display font-bold text-3xl text-accent">₹{plan.discountedPrice}</span>
              <span className="bg-red-100 text-red-700 text-xs font-bold px-2 py-1 rounded-full">{plan.savingsPercentage}% OFF</span>
            </div>
          </div>
        </div>

        {/* Features */}
        <div className="space-y-3 mb-8 flex-grow">
          <div className="flex items-center gap-3">
            <Check className="w-5 h-5 text-green-600 flex-shrink-0" />
            <span className="text-gray-700 text-sm">Fresh salads every week</span>
          </div>
          <div className="flex items-center gap-3">
            <Check className="w-5 h-5 text-green-600 flex-shrink-0" />
            <span className="text-gray-700 text-sm">Customizable ingredients</span>
          </div>
          <div className="flex items-center gap-3">
            <Check className="w-5 h-5 text-green-600 flex-shrink-0" />
            <span className="text-gray-700 text-sm">Free delivery</span>
          </div>
          <div className="flex items-center gap-3">
            <Check className="w-5 h-5 text-green-600 flex-shrink-0" />
            <span className="text-gray-700 text-sm">Nutritionist approved</span>
          </div>
        </div>

        <button className="w-full py-3 rounded-xl bg-accent hover:bg-accent/90 text-white font-bold text-lg shadow-lg shadow-accent/25 transition-all">
          {language === 'en' ? 'Subscribe Now' : 'आजच सदस्यता घ्या'}
        </button>
      </div>
    </div>
  );

  return (
    <div className="bg-gray-50/50 min-h-screen pb-20">
      {/* Header */}
      <div className="bg-primary pt-20 pb-24 px-4 text-center text-white relative overflow-hidden">
        <div className="absolute inset-0 bg-[url('https://images.unsplash.com/photo-1490645935967-10de6ba17061?q=80&w=2053&auto=format&fit=crop')] opacity-10 bg-cover bg-center" />
        <div className="relative z-10">
          <h1 className="text-4xl md:text-5xl font-display font-bold mb-4">{language === 'en' ? 'Subscription Plans' : 'सदस्यता योजना'}</h1>
          <p className="text-white/80 max-w-xl mx-auto text-lg">
            {language === 'en' 
              ? "Choose the perfect plan for your health goals. Fresh salads delivered every week." 
              : "आपल्या आरोग्याच्या लक्ष्यांसाठी परिपूर्ण योजना निवडा. दर आठवड्यात ताज्या सलाद."}
          </p>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 -mt-10 relative z-20">
        {/* Weekly Plans Section */}
        <div className="mb-16">
          <div className="text-center mb-12">
            <h2 className="text-3xl md:text-4xl font-display font-bold text-gray-900 mb-2">
              {language === 'en' ? 'Weekly Plans' : 'साप्ताहिक योजना'}
            </h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              {language === 'en' 
                ? "Perfect for busy professionals who want fresh, healthy salads delivered to their door." 
                : "व्यस्त व्यावसायिकांसाठी परिपूर्ण जो ताज्या, निरोगी सलाद घरी मिळवू इच्छितात."}
            </p>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
            {weeklyPlans.map((plan) => (
              <SubscriptionCard key={plan.id} plan={plan} />
            ))}
          </div>
        </div>

        {/* Monthly Plans Section */}
        <div>
          <div className="text-center mb-12">
            <h2 className="text-3xl md:text-4xl font-display font-bold text-gray-900 mb-2">
              {language === 'en' ? 'Monthly Plans' : 'मासिक योजना'}
            </h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              {language === 'en' 
                ? "Maximum savings and consistency for those committed to their health journey." 
                : "जे आपल्या आरोग्य यात्रेसाठी प्रतिबद्ध आहेत त्यांच्यासाठी कमाल बचत आणि सातत्य."}
            </p>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
            {monthlyPlans.map((plan) => (
              <SubscriptionCard key={plan.id} plan={plan} />
            ))}
          </div>
        </div>
      </div>
    </div>
  );
}
