import type { Express } from "express";
import type { Server } from "http";
import { storage } from "./storage";
import { api } from "@shared/routes";
import { z } from "zod";

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  // Salads endpoints
  app.get(api.salads.list.path, async (_req, res) => {
    const salads = await storage.getSalads();
    res.json(salads);
  });

  app.get(api.salads.get.path, async (req, res) => {
    const salad = await storage.getSalad(Number(req.params.id));
    if (!salad) {
      return res.status(404).json({ message: "Salad not found" });
    }
    res.json(salad);
  });

  // Subscriptions endpoints
  app.get(api.subscriptions.list.path, async (_req, res) => {
    const subscriptions = await storage.getSubscriptions();
    res.json(subscriptions);
  });

  app.get(api.subscriptions.get.path, async (req, res) => {
    const subscription = await storage.getSubscription(Number(req.params.id));
    if (!subscription) {
      return res.status(404).json({ message: "Subscription not found" });
    }
    res.json(subscription);
  });

  // Combos endpoints
  app.get(api.combos.list.path, async (_req, res) => {
    const combos = await storage.getCombos();
    res.json(combos);
  });

  app.get(api.combos.get.path, async (req, res) => {
    const combo = await storage.getCombo(Number(req.params.id));
    if (!combo) {
      return res.status(404).json({ message: "Combo not found" });
    }
    res.json(combo);
  });

  // Contact form submission
  app.post(api.contact.submit.path, async (req, res) => {
    try {
      const input = api.contact.submit.input.parse(req.body);
      const request = await storage.createContactRequest(input);
      res.status(201).json(request);
    } catch (err) {
      if (err instanceof z.ZodError) {
        return res.status(400).json({
          message: err.errors[0].message,
          field: err.errors[0].path.join('.'),
        });
      }
      throw err;
    }
  });

  app.get("/api/outlets", async (_req, res) => {
    const outlets = await storage.getOutlets();
    res.json(outlets);
  });

  await seedDatabase();

  return httpServer;
}

async function seedDatabase() {
  const existingOutlets = await storage.getOutlets();
  if (existingOutlets.length === 0) {
    const seedOutlets = [
      {
        name: "Daily Bowl - Thane West",
        marathiName: "डेली बाउल - ठाणे पश्चिम",
        address: "Gawand baug, kokanipada, Thane west 400 610",
        marathiAddress: "गावंड बाग, कोकणीपाडा, ठाणे पश्चिम ४०० ६१०",
        lat: "19.2452",
        lng: "72.9723",
        phone: "+91 9594620104",
      },
      {
        name: "Daily Bowl - Mulund",
        marathiName: "डेली बाउल - मुलुंड",
        address: "Shop 5, R-Mall, Mulund West, Mumbai 400080",
        marathiAddress: "दुकान ५, आर-मॉल, मुलुंड पश्चिम, मुंबई ४०००८०",
        lat: "19.1764",
        lng: "72.9525",
        phone: "+91 9594620105",
      }
    ];
    for (const outlet of seedOutlets) {
      await storage.createOutlet(outlet);
    }
  }

  const existingSalads = await storage.getSalads();
  if (existingSalads.length === 0) {
    const seedSalads = [
      {
        name: "Protein Power Bowl",
        marathiName: "प्रोटीन पॉवर बाउल",
        description: "A high-protein mix of chickpeas, kidney beans, and paneer with a tangy lemon dressing.",
        marathiDescription: "काबुली चणा, राजमा आणि पनीर यांचे लिंबाच्या रसासोबत हाय प्रोटीन मिश्रण.",
        price: 199,
        ingredients: ["Chickpeas", "Kidney Beans", "Paneer", "Lemon Dressing", "Cucumber", "Tomato"],
        calories: 350,
        protein: "22g",
        carbs: "45g",
        fat: "12g",
        isVeg: true,
        category: "Muscle Gain",
        imageUrl: "https://images.unsplash.com/photo-1546069901-ba9599a7e63c?w=800&q=80",
      },
      {
        name: "Green Detox Salad",
        marathiName: "ग्रीन डिटॉक्स सॅलड",
        description: "Fresh spinach, kale, avocado, and green apple for a refreshing detox.",
        marathiDescription: "ताजे पालक, केल, ॲव्होकॅडो आणि हिरवे सफरचंद यांचे डिटॉक्स मिश्रण.",
        price: 249,
        ingredients: ["Spinach", "Kale", "Avocado", "Green Apple", "Chia Seeds", "Honey Mustard Dressing"],
        calories: 220,
        protein: "5g",
        carbs: "18g",
        fat: "15g",
        isVeg: true,
        category: "Detox",
        imageUrl: "https://images.unsplash.com/photo-1512621776951-a57141f2eefd?w=800&q=80",
      },
      {
        name: "Exotic Fruit Bowl",
        marathiName: "एक्झॉटिक फ्रूट बाउल",
        description: "A colorful mix of seasonal fruits topped with seeds and honey.",
        marathiDescription: "हंगामी फळांचे बिया आणि मधासह रंगीबेरंगी मिश्रण.",
        price: 179,
        ingredients: ["Watermelon", "Papaya", "Pomegranate", "Kiwi", "Sunflower Seeds", "Honey"],
        calories: 180,
        protein: "3g",
        carbs: "40g",
        fat: "2g",
        isVeg: true,
        category: "Weight Loss",
        imageUrl: "https://images.unsplash.com/photo-1519996529931-28324d5a630e?w=800&q=80",
      },
      {
        name: "Paneer Tikka Salad",
        marathiName: "पनीर टिक्का सॅलड",
        description: "Grilled paneer cubes with roasted vegetables and mint chutney.",
        marathiDescription: "भाजलेल्या भाज्या आणि पुदिन्याच्या चटणीसह ग्रिल्ड पनीर.",
        price: 229,
        ingredients: ["Grilled Paneer", "Bell Peppers", "Onion", "Mint Chutney", "Lettuce"],
        calories: 310,
        protein: "18g",
        carbs: "12g",
        fat: "20g",
        isVeg: true,
        category: "Muscle Gain",
        imageUrl: "https://images.unsplash.com/photo-1567620905732-2d1ec7ab7445?w=800&q=80",
      },
      {
        name: "Sprouts & Moong Salad",
        marathiName: "स्प्राउट्स आणि मूग सॅलड",
        description: "Healthy sprouted moong with chopped veggies and chaat masala.",
        marathiDescription: "भाज्या आणि चाट मसाल्यासह मोड आलेले मूग.",
        price: 149,
        ingredients: ["Sprouted Moong", "Onion", "Tomato", "Coriander", "Chaat Masala", "Lemon Juice"],
        calories: 160,
        protein: "10g",
        carbs: "28g",
        fat: "1g",
        isVeg: true,
        category: "Weight Loss",
        imageUrl: "https://images.unsplash.com/photo-1547496502-ffa22d388bce?w=800&q=80",
      },
      {
        name: "Mediterranean Quinoa Bowl",
        marathiName: "मेडिटेरेनियन क्विनोआ बाउल",
        description: "Quinoa, olives, feta cheese, and cherry tomatoes with olive oil dressing.",
        marathiDescription: "क्विनोआ, ऑलिव्ह, फेटा चीज आणि चेरी टोमॅटो ऑलिव्ह ऑइल ड्रेसिंगसह.",
        price: 299,
        ingredients: ["Quinoa", "Olives", "Feta Cheese", "Cherry Tomatoes", "Cucumber", "Olive Oil"],
        calories: 340,
        protein: "12g",
        carbs: "42g",
        fat: "14g",
        isVeg: true,
        category: "Detox",
        imageUrl: "https://images.unsplash.com/photo-1511690656952-34342d5c2899?w=800&q=80",
      },
      {
        name: "Corn & Peanut Salsa",
        marathiName: "कॉर्न आणि शेंगदाणा सालसा",
        description: "Sweet corn and crunchy peanuts tossed with spicy salsa.",
        marathiDescription: "तिखट सालसासह स्वीट कॉर्न आणि शेंगदाणे.",
        price: 159,
        ingredients: ["Sweet Corn", "Roasted Peanuts", "Onion", "Tomato", "Green Chilli", "Lemon"],
        calories: 250,
        protein: "9g",
        carbs: "30g",
        fat: "11g",
        isVeg: true,
        category: "Weight Loss",
        imageUrl: "https://images.unsplash.com/photo-1529312266912-b33cf6227e24?w=800&q=80",
      },
      {
        name: "Classic Greek Salad",
        marathiName: "क्लासिक ग्रीक सॅलड",
        description: "Crispy lettuce, cucumber, olives, and feta cheese.",
        marathiDescription: "कुरकुरीत लेट्यूस, काकडी, ऑलिव्ह आणि फेटा चीज.",
        price: 269,
        ingredients: ["Lettuce", "Cucumber", "Black Olives", "Feta Cheese", "Red Onion", "Oregano"],
        calories: 210,
        protein: "6g",
        carbs: "10g",
        fat: "16g",
        isVeg: true,
        category: "Detox",
        imageUrl: "https://images.unsplash.com/photo-1540420773420-3366772f4999?w=800&q=80",
      },
      {
        name: "Beetroot & Walnut Salad",
        marathiName: "बीटरूट आणि अक्रोड सॅलड",
        description: "Roasted beetroot slices with crunchy walnuts and feta.",
        marathiDescription: "भाजलेले बीटरूटचे तुकडे, अक्रोड आणि फेटा.",
        price: 219,
        ingredients: ["Roasted Beetroot", "Walnuts", "Feta Cheese", "Arugula", "Balsamic Glaze"],
        calories: 230,
        protein: "7g",
        carbs: "15g",
        fat: "16g",
        isVeg: true,
        category: "Detox",
        imageUrl: "https://images.unsplash.com/photo-1505576399279-565b52d4ac71?w=800&q=80",
      },
      {
        name: "Tofu & Broccoli Stir Fry",
        marathiName: "टोफू आणि ब्रोकोली स्टिर फ्राय",
        description: "Grilled tofu and broccoli tossed in soy-garlic sauce.",
        marathiDescription: "सोया-लसूण सॉसमध्ये टोफू आणि ब्रोकोली.",
        price: 279,
        ingredients: ["Tofu", "Broccoli", "Bell Peppers", "Soy Sauce", "Garlic", "Sesame Seeds"],
        calories: 290,
        protein: "20g",
        carbs: "15g",
        fat: "18g",
        isVeg: true,
        category: "Muscle Gain",
        imageUrl: "https://images.unsplash.com/photo-1623428187969-5da2dcea5ebf?w=800&q=80",
      }
    ];

    for (const salad of seedSalads) {
      await storage.createSalad(salad);
    }
  }

  const existingSubscriptions = await storage.getSubscriptions();
  if (existingSubscriptions.length === 0) {
    const seedSubscriptions = [
      {
        name: "Weekly Starter",
        marathiName: "साप्ताहिक स्टार्टर",
        description: "Get 3 fresh salads every week with 15% savings",
        marathiDescription: "दर आठवड्यातून 3 ताजे सॅलड 15% सवयीत",
        type: "weekly",
        duration: 1,
        basePrice: 597,
        discountedPrice: 507,
        savingsPercentage: 15,
        saladsPerWeek: 3,
        imageUrl: "https://images.unsplash.com/photo-1546069901-ba9599a7e63c?w=800&q=80",
      },
      {
        name: "Weekly Premium",
        marathiName: "साप्ताहिक प्रीमियम",
        description: "Get 5 fresh salads every week with 20% savings",
        marathiDescription: "दर आठवड्यातून 5 ताजे सॅलड 20% सवयीत",
        type: "weekly",
        duration: 1,
        basePrice: 995,
        discountedPrice: 796,
        savingsPercentage: 20,
        saladsPerWeek: 5,
        imageUrl: "https://images.unsplash.com/photo-1512621776951-a57141f2eefd?w=800&q=80",
      },
      {
        name: "Monthly Starter",
        marathiName: "मासिक स्टार्टर",
        description: "Get 12 fresh salads every month with 25% savings",
        marathiDescription: "दर महिन्यातून 12 ताजे सॅलड 25% सवयीत",
        type: "monthly",
        duration: 1,
        basePrice: 2388,
        discountedPrice: 1791,
        savingsPercentage: 25,
        saladsPerWeek: 3,
        imageUrl: "https://images.unsplash.com/photo-1519996529931-28324d5a630e?w=800&q=80",
      },
      {
        name: "Monthly Premium",
        marathiName: "मासिक प्रीमियम",
        description: "Get 20 fresh salads every month with 30% savings",
        marathiDescription: "दर महिन्यातून 20 ताजे सॅलड 30% सवयीत",
        type: "monthly",
        duration: 1,
        basePrice: 3980,
        discountedPrice: 2786,
        savingsPercentage: 30,
        saladsPerWeek: 5,
        imageUrl: "https://images.unsplash.com/photo-1540420773420-3366772f4999?w=800&q=80",
      },
    ];

    for (const subscription of seedSubscriptions) {
      await storage.createSubscription(subscription);
    }
  }

  const existingCombos = await storage.getCombos();
  if (existingCombos.length === 0) {
    const seedCombos = [
      {
        name: "Power & Detox Duo",
        marathiName: "पॉवर आणि डिटॉक्स ड्यू",
        description: "Protein Power Bowl + Green Detox Salad",
        marathiDescription: "प्रोटीन पॉवर बाउल + ग्रीन डिटॉक्स सॅलड",
        saladIds: ["1", "2"],
        basePrice: 448,
        comboPrice: 379,
        discount: 15,
        servings: 2,
        imageUrl: "https://images.unsplash.com/photo-1546069901-ba9599a7e63c?w=800&q=80",
      },
      {
        name: "Triple Health Pack",
        marathiName: "ट्रिपल हेल्थ पॅक",
        description: "Protein Power + Green Detox + Paneer Tikka",
        marathiDescription: "प्रोटीन पॉवर + ग्रीन डिटॉक्स + पनीर टिक्का",
        saladIds: ["1", "2", "4"],
        basePrice: 677,
        comboPrice: 542,
        discount: 20,
        servings: 3,
        imageUrl: "https://images.unsplash.com/photo-1512621776951-a57141f2eefd?w=800&q=80",
      },
      {
        name: "Weight Loss Combo",
        marathiName: "वजन कमी करण्याचा कॉम्बो",
        description: "Exotic Fruit + Sprouts & Moong + Corn & Peanut",
        marathiDescription: "एक्झॉटिक फ्रूट + स्प्राउट्स & मूग + कॉर्न & शेंगदाणा",
        saladIds: ["3", "5", "7"],
        basePrice: 488,
        comboPrice: 390,
        discount: 20,
        servings: 3,
        imageUrl: "https://images.unsplash.com/photo-1519996529931-28324d5a630e?w=800&q=80",
      },
      {
        name: "Mediterranean Bliss",
        marathiName: "मेडिटेरेनियन ब्लिस",
        description: "Mediterranean Quinoa + Greek Salad + Beetroot Walnut",
        marathiDescription: "मेडिटेरेनियन क्विनोआ + ग्रीक सॅलड + बीटरूट अक्रोड",
        saladIds: ["6", "8", "9"],
        basePrice: 788,
        comboPrice: 630,
        discount: 20,
        servings: 3,
        imageUrl: "https://images.unsplash.com/photo-1511690656952-34342d5c2899?w=800&q=80",
      },
    ];

    for (const combo of seedCombos) {
      await storage.createCombo(combo);
    }
  }
}
